package com.caysn.autoreplyprint.cmprint;

import android.graphics.Bitmap;

import com.caysn.autoreplyprint.AutoReplyPrint;
import com.sun.jna.Pointer;

/**
 * 通用打印类，所有机型均可调用该类中的函数。
 * 主要有查询打印机实时状态，打印自检页，打印图片等功能。
 */
public class CMPrintCommon {

    // 连续纸
    public static final int CMPaperTypeSerialPaper = 1;
    // 间隙纸
    public static final int CMPaperTypeGapPaper = 2;
    // 标记纸
    public static final int CMPaperTypeMarkPaper = 3;

    // 切刀不切
    public static final int CMCutterNoCutPaper = 0;
    // 切刀全切
    public static final int CMCutterFullCutPaper = 1;
    // 切刀半切
    public static final int CMCutterHalfCutPaper = 2;

    // 查询打印机实时状态，如果打印机未连接，查询失败，则返回null。
    public static CMPrinterRTStatus queryPrinterRTStatus(CMPrinterConnector printerConnector) {
        Pointer printerHandle = printerConnector.getCurrentPrinterHandle();
        int rt_status = AutoReplyPrint.INSTANCE.CP_Pos_QueryRTStatus(printerHandle, 3000);
        if (rt_status != 0) {
            return new CMPrinterRTStatus(rt_status);
        }
        return null;
    }

    // 打印自检页，自检页上会显示打印机软件版本，还有其他有用的信息，便于出现问题时，快速排查问题。
    public static boolean printSelfTestPage(CMPrinterConnector printerConnector) {
        Pointer printerHandle = printerConnector.getCurrentPrinterHandle();
        return AutoReplyPrint.INSTANCE.CP_Pos_PrintSelfTestPage(printerHandle);
    }

    /**
     * 打印图片，并指定打印图片前后的相关动作
     *
     * @param printerConnector  一个打印机连接
     * @param bitmap            要打印的图片，注意图片宽度不要超过了打印机可打印区域，超了可能会乱码
     * @param paperType         指定纸张类型，如果指定了标签纸或黑标纸，则打印完会自动定位到缝隙处
     *                          1 连续纸
     *                          2 标签纸
     *                          3 黑标纸
     * @param kickDrawer        打印前是否开钱箱，由于打印需要时间，如果打印后开钱箱，会影响效率，一般都是打印前开钱箱
     * @param feedDots          打印完图片，额外进纸点行，主要是多走一点纸，便于撕纸
     * @param cutPaper          打印完图片，进完纸之后，进行切刀，有切刀的机器，用切刀切纸，不需要专门去撕纸，0是不切纸，1是全切，2是半切
     * @param waitPrintFinished 是否等待打印完成。
     *                          取值0表示数据发送完成直接返回，不等待打印完成；
     *                          取值大于0表示，最多等待对应的毫秒时间直到打印完成，如果在超时时间内打印完成，则返回true，如果超时到了还没打完，返回false。
     * @return 数据写入成功，返回true，数据写入失败，返回false。如果设置了等待打印完成，则超时时间内打印完成，才返回true，否则返回false。
     */
    public static boolean printBitmap(CMPrinterConnector printerConnector, Bitmap bitmap, int paperType, boolean kickDrawer, int feedDots, int cutPaper, int waitPrintFinished) {
        boolean printResult = false;

        Pointer printerHandle = printerConnector.getCurrentPrinterHandle();

        if (kickDrawer) {
            AutoReplyPrint.INSTANCE.CP_Pos_KickOutDrawer(printerHandle, 0, 100, 100);
            AutoReplyPrint.INSTANCE.CP_Pos_KickOutDrawer(printerHandle, 1, 100, 100);
        }

        if (paperType == CMPaperTypeGapPaper) {
            // 标签纸
            byte[] cmd = new byte[]{0x1a, 0x0c, (byte) 0xff};
            AutoReplyPrint.INSTANCE.CP_Port_Write(printerHandle, cmd, cmd.length, 10000);
        } else if (paperType == CMPaperTypeMarkPaper) {
            // 黑标纸
        } else {
            // 连续纸
        }

        printResult = AutoReplyPrint.CP_Pos_PrintRasterImageFromData_Helper.PrintRasterImageFromBitmap(printerHandle, bitmap.getWidth(), bitmap.getHeight(), bitmap, AutoReplyPrint.CP_ImageBinarizationMethod_ErrorDiffusion, AutoReplyPrint.CP_ImageCompressionMethod_Level1);

        if (paperType == CMPaperTypeGapPaper) {
            // 标签纸
            byte[] cmd = new byte[]{0x1a, 0x0c, 0x00};
            AutoReplyPrint.INSTANCE.CP_Port_Write(printerHandle, cmd, cmd.length, 10000);
        } else if (paperType == CMPaperTypeMarkPaper) {
            // 黑标纸
            AutoReplyPrint.INSTANCE.CP_BlackMark_FindNextBlackMark(printerHandle);
        } else {
            // 连续纸
        }

        if (feedDots > 0) {
            AutoReplyPrint.INSTANCE.CP_Pos_FeedDot(printerHandle, feedDots);
        }
        if (cutPaper == CMCutterFullCutPaper) {
            AutoReplyPrint.INSTANCE.CP_Pos_FullCutPaper(printerHandle);
        } else if (cutPaper == CMCutterHalfCutPaper) {
            AutoReplyPrint.INSTANCE.CP_Pos_HalfCutPaper(printerHandle);
        }

        if (waitPrintFinished > 0) {
            printResult = AutoReplyPrint.INSTANCE.CP_Pos_QueryPrintResult(printerHandle, waitPrintFinished);
        }

        return printResult;
    }

}
