package com.caysn.autoreplyprint.caprint;

import android.graphics.Bitmap;

import com.caysn.autoreplyprint.AutoReplyPrint;
import com.caysn.autoreplyprint.NZPrivate;
import com.sun.jna.Pointer;
import com.sun.jna.ptr.IntByReference;
import com.sun.jna.ptr.LongByReference;

/**
 * 通用打印类，所有机型均可调用该类中的函数。
 * 主要有获取打印机状态，获取打印机分辨率，打印自检页，打印图片等功能。
 */
public class CAPrintCommon {

    // 连续纸
    public static final int CAPaperTypeSerialPaper = 1;
    // 间隙纸
    public static final int CAPaperTypeGapPaper = 2;
    // 标记纸
    public static final int CAPaperTypeMarkPaper = 3;

    // 切刀不切
    public static final int CACutterNoCutPaper = 0;
    // 切刀全切
    public static final int CACutterFullCutPaper = 1;
    // 切刀半切
    public static final int CACutterHalfCutPaper = 2;

    // 获取打印机状态，如果打印机未连接，获取失败，则返回null。
    public static CAPrinterStatus getPrinterStatus(CAPrinterConnector printerConnector) {
        Pointer printerHandle = printerConnector.getCurrentPrinterHandle();
        LongByReference printer_error_status = new LongByReference();
        LongByReference printer_info_status = new LongByReference();
        LongByReference timestamp_ms = new LongByReference();
        if (AutoReplyPrint.INSTANCE.CP_Printer_GetPrinterStatusInfo(printerHandle, printer_error_status, printer_info_status, timestamp_ms)) {
            return new CAPrinterStatus(printer_error_status.getValue(), printer_info_status.getValue());
        }
        return null;
    }

    // 获取打印机分辨率，如果打印机未连接，获取失败，则返回null。
    public static CAPrinterResolution getPrinterResolution(CAPrinterConnector printerConnector) {
        Pointer printerHandle = printerConnector.getCurrentPrinterHandle();
        IntByReference width_mm = new IntByReference();
        IntByReference height_mm = new IntByReference();
        IntByReference dots_per_mm = new IntByReference();
        if (AutoReplyPrint.INSTANCE.CP_Printer_GetPrinterResolutionInfo(printerHandle, width_mm, height_mm, dots_per_mm)) {
            return new CAPrinterResolution(width_mm.getValue(), height_mm.getValue(), dots_per_mm.getValue());
        }
        return null;
    }

    // 打印自检页，自检页上会显示打印机软件版本，还有其他有用的信息，便于出现问题时，快速排查问题。
    public static boolean printSelfTestPage(CAPrinterConnector printerConnector) {
        Pointer printerHandle = printerConnector.getCurrentPrinterHandle();
        return AutoReplyPrint.INSTANCE.CP_Pos_PrintSelfTestPage(printerHandle);
    }

    private static byte[] getBitmapPrintCmd(CAPrinterConnector printerConnector, Bitmap bitmap, int binaryzation_method, int compression_method, int paperType, int printAlignment, int printDensity, boolean kickDrawer, double feedPaper, int cutPaper) {
        byte[] cmd_data = null;

        Pointer h = NZPrivate.INSTANCE.CP_Port_OpenMemoryBuffer(0x100000);
        if (AutoReplyPrint.INSTANCE.CP_Port_IsOpened(h)) {

            CAPrinterResolution printerResolution = CAPrintCommon.getPrinterResolution(printerConnector);

            if ((printDensity >= 0) && (printDensity <= 15)) {
                AutoReplyPrint.INSTANCE.CP_Pos_SetPrintDensity(h, printDensity);
            }

            if (kickDrawer) {
                AutoReplyPrint.INSTANCE.CP_Pos_KickOutDrawer(h, 0, 100, 100);
                AutoReplyPrint.INSTANCE.CP_Pos_KickOutDrawer(h, 1, 100, 100);
            }

            if (printerResolution != null) {
                AutoReplyPrint.INSTANCE.CP_Pos_SetPrintAreaLeftMargin(h, 0);
                AutoReplyPrint.INSTANCE.CP_Pos_SetPrintAreaWidth(h, printerResolution.getWidthMM() * printerResolution.getDotsPerMM());
            }

            AutoReplyPrint.INSTANCE.CP_Pos_SetAlignment(h, printAlignment);

            if (paperType == CAPaperTypeGapPaper) {
                // 标签纸
                byte[] cmd = new byte[]{0x1a, 0x0c, (byte) 0xff};
                AutoReplyPrint.INSTANCE.CP_Port_Write(h, cmd, cmd.length, 10000);
            } else if (paperType == CAPaperTypeMarkPaper) {
                // 黑标纸
            } else {
                // 连续纸
            }

            AutoReplyPrint.CP_Pos_PrintRasterImageFromData_Helper.PrintRasterImageFromBitmap(h, bitmap.getWidth(), bitmap.getHeight(), bitmap, binaryzation_method, compression_method);

            if (paperType == CAPaperTypeGapPaper) {
                // 标签纸
                byte[] cmd = new byte[]{0x1a, 0x0c, 0x00};
                AutoReplyPrint.INSTANCE.CP_Port_Write(h, cmd, cmd.length, 10000);
            } else if (paperType == CAPaperTypeMarkPaper) {
                // 黑标纸
                AutoReplyPrint.INSTANCE.CP_BlackMark_FindNextBlackMark(h);
            } else {
                // 连续纸
            }

            if (feedPaper > 0) {
                int dots_per_mm = 8;
                if (printerResolution != null) {
                    dots_per_mm = printerResolution.getDotsPerMM();
                }
                int feedDots = (int) Math.round(feedPaper * dots_per_mm);
                AutoReplyPrint.INSTANCE.CP_Pos_FeedDot(h, feedDots);
            }
            if (cutPaper == CACutterFullCutPaper) {
                AutoReplyPrint.INSTANCE.CP_Pos_FullCutPaper(h);
            } else if (cutPaper == CACutterHalfCutPaper) {
                AutoReplyPrint.INSTANCE.CP_Pos_HalfCutPaper(h);
            }

            cmd_data = NZPrivate.CP_Port_GetMemoryBufferData_Helper.GetMemoryBufferData(h);

            AutoReplyPrint.INSTANCE.CP_Port_Close(h);
        }

        return cmd_data;
    }

    private static byte[] byteArraysToBytes(byte[][] data) {
        int length = 0;
        for (int i = 0; i < data.length; i++)
            length += data[i].length;
        byte[] send = new byte[length];
        int k = 0;
        for (int i = 0; i < data.length; i++)
            for (int j = 0; j < data[i].length; j++)
                send[k++] = data[i][j];
        return send;
    }

    // 获取蓝牙极限速度，单位是字节每秒
    // 不知道为什么，用P80D打印，明显感觉卡顿。
    private static int getSppMaxSpeedInBytesPerSecond() {
        return 20 * 1024;
    }

    /**
     * 打印图片，并指定打印图片前后的相关动作
     *
     * @param printerConnector    一个打印机连接
     * @param bitmap              要打印的图片，注意图片宽度不要超过了打印机可打印区域，超了可能会乱码
     * @param binaryzation_method 图片二值化算法
     *                            0 抖动算法
     *                            1 阈值算法
     *                            2 误差扩散法
     * @param compression_method  图片压缩算法
     *                            0 不压缩
     *                            1 压缩算法1
     *                            2 压缩算法2
     * @param paperType           指定纸张类型，如果指定了标签纸或黑标纸，则打印完会自动定位到缝隙处
     *                            1 连续纸
     *                            2 标签纸
     *                            3 黑标纸
     * @param printAlignment      指定对齐方式，如果图片宽度小于可打印宽度，不指定的话，可能打偏
     *                            0 左对齐
     *                            1 居中对齐
     *                            2 右对齐
     * @param printSpeed          指定打印速度，0表示不指定打印速度，大于零表示指定打印速度为多少mm/s。（蓝牙和串口打印速度会根据数据内容自动调整）
     * @param printDensity        指定打印浓度，浓度范围[0-15]。如果设置为-1，则不指定打印浓度。
     * @param kickDrawer          打印前是否开钱箱，由于打印需要时间，如果打印后开钱箱，会影响效率，一般都是打印前开钱箱
     * @param feedPaper           打印完图片，额外进纸毫米，主要是多走一点纸，便于撕纸
     * @param cutPaper            打印完图片，进完纸之后，进行切刀，有切刀的机器，用切刀切纸，不需要专门去撕纸，0是不切纸，1是全切，2是半切
     * @param waitPrintFinished   是否等待打印完成。
     *                            取值0表示数据发送完成直接返回，不等待打印完成；
     *                            取值大于0表示，最多等待对应的毫秒时间直到打印完成，如果在超时时间内打印完成，则返回true，如果超时到了还没打完，返回false。
     * @return 数据写入成功，返回true，数据写入失败，返回false。如果设置了等待打印完成，则超时时间内打印完成，才返回true，否则返回false。
     */
    public static boolean printBitmap(CAPrinterConnector printerConnector, Bitmap bitmap, int binaryzation_method, int compression_method, int paperType, int printAlignment, int printSpeed, int printDensity, boolean kickDrawer, double feedPaper, int cutPaper, int waitPrintFinished) {
        CAPrintLogger.debug("CAPrintCommon printBitmap started");

        CAPrintResult caPrintResult = new CAPrintResult();

        Pointer printerHandle = printerConnector.getCurrentPrinterHandle();
        CAPrinterDevice printerDevice = printerConnector.getCurrentPrinterDevice();
        CAPrinterResolution printerResolution = CAPrintCommon.getPrinterResolution(printerConnector);

        // 计算图片高度
        if ((printerResolution != null) && (printerResolution.getDotsPerMM() > 0)) {
            double imageHeightMM = (double) bitmap.getHeight() / printerResolution.getDotsPerMM();
            caPrintResult.sourceImageHeight = imageHeightMM;
        }

        // 获取打印命令
        byte[] cmd_data = getBitmapPrintCmd(printerConnector, bitmap, binaryzation_method, compression_method, paperType, printAlignment, printDensity, kickDrawer, feedPaper, cutPaper);

        // 设置打印速度
        if (printSpeed > 0) {
            // 如果是蓝牙打印，则根据蓝牙速度，计算极限速度，临时设置打印机速度
            if ((printerDevice != null) && (cmd_data != null)) {
                int dataTransferSpeedInBytesPerSecond = 0;
                if (printerDevice.port_type.compareTo(CAPrinterDevice.PrinterDevicePortTypeBtSpp) == 0) {
                    //蓝牙接口数据传输速度
                    dataTransferSpeedInBytesPerSecond = getSppMaxSpeedInBytesPerSecond();
                } else if (printerDevice.port_type.compareTo(CAPrinterDevice.PrinterDevicePortTypeCom) == 0) {
                    //串口接口数据传输速度
                    dataTransferSpeedInBytesPerSecond = (int) (printerDevice.comBaudrate / 10);
                }
                if (dataTransferSpeedInBytesPerSecond > 0) {
                    double estimatedTransferTimeSeconds = ((double) cmd_data.length / (double) dataTransferSpeedInBytesPerSecond);
                    if (estimatedTransferTimeSeconds > 0) {
                        int estimatedPrintSpeed = (int) (Math.round(caPrintResult.sourceImageHeight / estimatedTransferTimeSeconds));
                        if (estimatedPrintSpeed < 30)
                            estimatedPrintSpeed = 30;
                        if (estimatedPrintSpeed > 300)
                            estimatedPrintSpeed = 300;
                        if (printSpeed > estimatedPrintSpeed)
                            printSpeed = estimatedPrintSpeed;
                    }
                }
            }

            byte[] setPrintSpeedCmd = new byte[]{0x1F, 0x28, 0x73, 0x02, 0x00, (byte) (printSpeed & 0xff), (byte) ((printSpeed >> 8) & 0xff)};
            cmd_data = byteArraysToBytes(new byte[][]{setPrintSpeedCmd, cmd_data});

            caPrintResult.settedPrintSpeed = printSpeed;
        }

        // 发送打印数据
        if (cmd_data != null) {
            long beginTime = System.currentTimeMillis();
            CAPrintLogger.debug("CAPrintCommon printBitmap CP_Port_Write started");
            caPrintResult.printResult = (AutoReplyPrint.INSTANCE.CP_Port_Write(printerHandle, cmd_data, cmd_data.length, 180 * 1000) == cmd_data.length);
            CAPrintLogger.debug("CAPrintCommon printBitmap CP_Port_Write finished");
            long endTime = System.currentTimeMillis();
            double used_second = (endTime - beginTime) / 1000.0;

            caPrintResult.writeDataSize = cmd_data.length;
            caPrintResult.writeTimeUsed = used_second * 1000;
            if ((caPrintResult.writeDataSize > 0) && (caPrintResult.writeTimeUsed > 0)) {
                caPrintResult.writeDataSpeed = (double) (caPrintResult.writeDataSize) / caPrintResult.writeTimeUsed / 1024 * 1000;
            }
        }

        // 等待打印完成
        if (waitPrintFinished > 0) {
            CAPrintLogger.debug("CAPrintCommon printBitmap CP_Pos_QueryPrintResult started");
            caPrintResult.printResult = AutoReplyPrint.INSTANCE.CP_Pos_QueryPrintResult(printerHandle, waitPrintFinished);
            CAPrintLogger.debug("CAPrintCommon printBitmap CP_Pos_QueryPrintResult started");
        }

        CAPrintLogger.debug("CAPrintCommon printBitmap finished " + (caPrintResult.printResult ? "success" : "failed"));

        return caPrintResult.printResult;
    }

    // 将字节数组直接发送到打印机。主要用于发一些小指令，比如设置各种参数的指令等等。
    public static int writeBuffer(CAPrinterConnector printerConnector, byte[] buffer) {
        Pointer printerHandle = printerConnector.getCurrentPrinterHandle();
        return AutoReplyPrint.INSTANCE.CP_Port_Write(printerHandle, buffer, buffer.length, 10000);
    }

    // 查询打印机电池电量。范围在0-100之间。返回-1表示查询失败。
    public static int queryPrinterBatteryLevel(CAPrinterConnector printerConnector) {
        Pointer printerHandle = printerConnector.getCurrentPrinterHandle();
        return AutoReplyPrint.INSTANCE.CP_Proto_QueryBatteryLevel(printerHandle, 3000);
    }

    // 查询打印机序列号
    public static String queryPrinterSerialNumber(CAPrinterConnector printerConnector) {
        Pointer printerHandle = printerConnector.getCurrentPrinterHandle();
        return AutoReplyPrint.CP_Proto_QuerySerialNumber_Helper.QuerySerialNumber(printerHandle, 3000);
    }

    // 设置系统名称和序列号
    public static boolean CA_Proto_SetSystemNameAndSerialNumber(CAPrinterConnector printerConnector, String systemName, String serialNumber) {
        Pointer printerHandle = printerConnector.getCurrentPrinterHandle();
        return AutoReplyPrint.INSTANCE.CP_Proto_SetSystemNameAndSerialNumber(printerHandle, systemName, serialNumber);
    }

    // 设置蓝牙名称和密码
    public static boolean CA_Proto_SetBluetoothNameAndPassword(CAPrinterConnector printerConnector, String bluetoothName, String bluetoothPassword) {
        Pointer printerHandle = printerConnector.getCurrentPrinterHandle();
        return AutoReplyPrint.INSTANCE.CP_Proto_SetBluetoothNameAndPassword(printerHandle, bluetoothName, bluetoothPassword);
    }

    // 设置基础参数，包括语言、波特率、浓度等。
    public static boolean CA_Proto_SetPTPBasicParameters(CAPrinterConnector printerConnector, int baudrate, int codepage, int density, int asciiFontType, int lineFeed, int idleTime, int powerOffTime, int maxFeedLength, int pageLength) {
        Pointer printerHandle = printerConnector.getCurrentPrinterHandle();
        return AutoReplyPrint.INSTANCE.CP_Proto_SetPTPBasicParameters(printerHandle, baudrate, codepage, density, asciiFontType, lineFeed, idleTime, powerOffTime, maxFeedLength, pageLength);
    }

    // 设置打印速度
    public static boolean CA_Settings_Hardware_SetPrintSpeed(CAPrinterConnector printerConnector, int nSpeed) {
        Pointer printerHandle = printerConnector.getCurrentPrinterHandle();
        return AutoReplyPrint.INSTANCE.CP_Settings_Hardware_SetPrintSpeed(printerHandle, nSpeed);
    }

}

/**
 * 接口尽量保持简洁，不要引入太多无用的接口。
 * 基于此考虑
 * 1 不加入读数据的接口
 * 2 加入写数据的接口，并且接口参数简化，去掉超时，仅考虑写入少量数据的情况
 */
