package com.caysn.autoreplyprint.cmprint;

import com.caysn.autoreplyprint.AutoReplyPrint;
import com.sun.jna.Pointer;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.locks.ReentrantLock;

/**
 * 打印机连接类，用来连接和断开一个打印机设备。
 * 如果注册连接状态回调，则可以在连接状态变化时，获得通知。
 * 若设备端关闭，此处也可以及时知道，并触发连接状态变化回调。
 * 一般USB数据线拔出或打印机关闭，可立刻知道。
 * 蓝牙2.0打印机关闭，大约2s可以知道。
 * TCP网络打印机关闭，大约30s可以知道。
 */
public class CMPrinterConnector {

    private static final String TAG = "CMPrinterConnector";

    public static final int CMPrinterConnector_ConnectionStatus_Disconnected = 0;
    public static final int CMPrinterConnector_ConnectionStatus_Connecting = 1;
    public static final int CMPrinterConnector_ConnectionStatus_Connected = 2;
    public static final int CMPrinterConnector_ConnectionStatus_Disconnecting = 3;

    private CMPrinterConnectPortInfo currentPortInfo = null;
    private Pointer currentPrinterHandle = Pointer.NULL;
    private int currentConnectionStatus = CMPrinterConnector_ConnectionStatus_Disconnected;
    private ReentrantLock connectionLock = new ReentrantLock();

    private AutoReplyPrint.CP_OnPortClosedEvent_Callback m_closed_callback = new AutoReplyPrint.CP_OnPortClosedEvent_Callback() {
        @Override
        public void CP_OnPortClosedEvent(Pointer handle, Pointer private_data) {
            if (!handle.equals(currentPrinterHandle))
                return;
            disconnectCurrentPrinter();
        }
    };

    public CMPrinterConnector() {
        AutoReplyPrint.INSTANCE.CP_Port_AddOnPortClosedEvent(m_closed_callback, Pointer.NULL);
    }

    protected void finalize() {
        AutoReplyPrint.INSTANCE.CP_Port_RemoveOnPortClosedEvent(m_closed_callback);
        disconnectCurrentPrinter();
    }


    private boolean isConnectThreadRunning = false;

    // 启动一个子线程去连接打印机设备（异步），连接结果需要通过 连接状态变化回调接口 获取。
    // 返回值仅表示本次是否启动子线程。
    public boolean connectPrinterAsync(final CMPrinterConnectPortInfo portInfo) {
        if (isConnectThreadRunning)
            return false;
        isConnectThreadRunning = true;
        new Thread(new Runnable() {
            @Override
            public void run() {
                connectPrinterSync(portInfo);
                isConnectThreadRunning = false;
            }
        }).start();
        return true;
    }

    // 直接连接打印机设备（同步），返回连接结果。
    public boolean connectPrinterSync(CMPrinterConnectPortInfo portInfo) {
        try {
            connectionLock.lock();
        } catch (Throwable tr) {
            tr.printStackTrace();
        }

        try {
            if ((currentConnectionStatus == CMPrinterConnector_ConnectionStatus_Connected) && (!CMPrinterConnectPortInfo.isSamePort(currentPortInfo, portInfo))) {
                disconnectCurrentPrinter();
            }
        } catch (Throwable tr) {
            tr.printStackTrace();
        }

        if (currentConnectionStatus != CMPrinterConnector_ConnectionStatus_Connected) {
            try {
                currentPortInfo = portInfo;
                currentConnectionStatus = CMPrinterConnector_ConnectionStatus_Connecting;
                triggerConnectionStatusChangedEvent();
            } catch (Throwable tr) {
                tr.printStackTrace();
            }

            try {
                int autoreplymode = 0;
                if (CMPrinterConnectPortInfo.CMPrinterConnectComPortInfo.class.isInstance(portInfo)) {
                    CMPrinterConnectPortInfo.CMPrinterConnectComPortInfo p = (CMPrinterConnectPortInfo.CMPrinterConnectComPortInfo) portInfo;
                    currentPrinterHandle = AutoReplyPrint.INSTANCE.CP_Port_OpenCom(p.com_name, p.com_baudrate, p.com_databits, p.com_parity, p.com_stopbits, p.com_flowcontrol, autoreplymode);
                } else if (CMPrinterConnectPortInfo.CMPrinterConnectUsbPortInfo.class.isInstance(portInfo)) {
                    CMPrinterConnectPortInfo.CMPrinterConnectUsbPortInfo p = (CMPrinterConnectPortInfo.CMPrinterConnectUsbPortInfo) portInfo;
                    currentPrinterHandle = AutoReplyPrint.INSTANCE.CP_Port_OpenUsb(p.usb_path, autoreplymode);
                } else if (CMPrinterConnectPortInfo.CMPrinterConnectNetTcpPortInfo.class.isInstance(portInfo)) {
                    CMPrinterConnectPortInfo.CMPrinterConnectNetTcpPortInfo p = (CMPrinterConnectPortInfo.CMPrinterConnectNetTcpPortInfo) portInfo;
                    currentPrinterHandle = AutoReplyPrint.INSTANCE.CP_Port_OpenTcp(p.local_ip, p.dest_ip, (short) p.dest_port, 5000, autoreplymode);
                } else if (CMPrinterConnectPortInfo.CMPrinterConnectNetUdpPortInfo.class.isInstance(portInfo)) {
                    currentPrinterHandle = Pointer.NULL;
                } else if (CMPrinterConnectPortInfo.CMPrinterConnectBtSppPortInfo.class.isInstance(portInfo)) {
                    CMPrinterConnectPortInfo.CMPrinterConnectBtSppPortInfo p = (CMPrinterConnectPortInfo.CMPrinterConnectBtSppPortInfo) portInfo;
                    currentPrinterHandle = AutoReplyPrint.INSTANCE.CP_Port_OpenBtSpp(p.bluetooth_address, autoreplymode);
                } else if (CMPrinterConnectPortInfo.CMPrinterConnectBtBlePortInfo.class.isInstance(portInfo)) {
                    CMPrinterConnectPortInfo.CMPrinterConnectBtBlePortInfo p = (CMPrinterConnectPortInfo.CMPrinterConnectBtBlePortInfo) portInfo;
                    currentPrinterHandle = AutoReplyPrint.INSTANCE.CP_Port_OpenBtBle(p.bluetooth_address, autoreplymode);
                } else {
                    currentPrinterHandle = Pointer.NULL;
                }
            } catch (Throwable tr) {
                tr.printStackTrace();
            }

            try {
                if (currentPrinterHandle != Pointer.NULL) {
                    currentConnectionStatus = CMPrinterConnector_ConnectionStatus_Connected;
                } else {
                    currentConnectionStatus = CMPrinterConnector_ConnectionStatus_Disconnected;
                }
                triggerConnectionStatusChangedEvent();
            } catch (Throwable tr) {
                tr.printStackTrace();
            }
        }

        try {
            connectionLock.unlock();
        } catch (Throwable tr) {
            tr.printStackTrace();
        }

        return (currentConnectionStatus == CMPrinterConnector_ConnectionStatus_Connected);
    }

    // 断开打印机连接
    public void disconnectCurrentPrinter() {
        if (currentPrinterHandle != Pointer.NULL) {
            try {
                connectionLock.lock();
            } catch (Throwable tr) {
                tr.printStackTrace();
            }

            if (currentPrinterHandle != Pointer.NULL) {
                try {
                    currentConnectionStatus = CMPrinterConnector_ConnectionStatus_Disconnecting;
                    triggerConnectionStatusChangedEvent();
                } catch (Throwable tr) {
                    tr.printStackTrace();
                }

                try {
                    AutoReplyPrint.INSTANCE.CP_Port_Close(currentPrinterHandle);
                    currentPrinterHandle = Pointer.NULL;
                } catch (Throwable tr) {
                    tr.printStackTrace();
                }

                try {
                    currentConnectionStatus = CMPrinterConnector_ConnectionStatus_Disconnected;
                    triggerConnectionStatusChangedEvent();
                } catch (Throwable tr) {
                    tr.printStackTrace();
                }
            }

            try {
                connectionLock.unlock();
            } catch (Throwable tr) {
                tr.printStackTrace();
            }
        }
    }


    // 获取当前打印机连接端口信息
    public CMPrinterConnectPortInfo getCurrentPortInfo() {
        return currentPortInfo;
    }

    // 获取当前打印机连接句柄
    public Pointer getCurrentPrinterHandle() {
        return currentPrinterHandle;
    }

    // 获取当前连接状态
    public int getCurrentConnectionStatus() {
        return currentConnectionStatus;
    }

    // 是否正在连接打印机
    public boolean isCurrentConnectingPrinter() {
        return currentConnectionStatus == CMPrinterConnector_ConnectionStatus_Connecting;
    }

    // 是否已连接打印机
    public boolean isCurrentConnectedPrinter() {
        return currentConnectionStatus == CMPrinterConnector_ConnectionStatus_Connected;
    }

    // 是否正在断开打印机
    public boolean isCurrentDisconnectingPrinter() {
        return currentConnectionStatus == CMPrinterConnector_ConnectionStatus_Disconnecting;
    }

    // 是否已断开打印机
    public boolean isCurrentDisconnectedPrinter() {
        return currentConnectionStatus == CMPrinterConnector_ConnectionStatus_Disconnected;
    }


    // 连接状态变化回调接口
    public interface ConnectionStatusChangedInterface {
        void onConnectionStatusChanged();
    }

    private List<ConnectionStatusChangedInterface> m_connectionStatusChangedInterfaceList = new ArrayList<ConnectionStatusChangedInterface>();

    // 注册连接状态变化回调
    public synchronized void registerConnectionStatusChangedEvent(ConnectionStatusChangedInterface event) {
        if (!m_connectionStatusChangedInterfaceList.contains(event))
            m_connectionStatusChangedInterfaceList.add(event);
    }

    // 移除连接状态变化回调
    public synchronized void unregisterConnectionStatusChangedEvent(ConnectionStatusChangedInterface event) {
        if (m_connectionStatusChangedInterfaceList.contains(event))
            m_connectionStatusChangedInterfaceList.remove(event);
    }

    private void triggerConnectionStatusChangedEvent() {
        for (ConnectionStatusChangedInterface event : m_connectionStatusChangedInterfaceList) {
            event.onConnectionStatusChanged();
        }
    }

}
