package com.caysn.autoreplyprint.caprint;

import com.caysn.autoreplyprint.AutoReplyPrint;
import com.sun.jna.Pointer;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.locks.ReentrantLock;

/**
 * 打印机连接类，用来连接和断开一个打印机设备。
 * 如果注册连接状态回调，则可以在连接状态变化时，获得通知。
 * 若设备端关闭，此处也可以及时知道，并触发连接状态变化回调。
 * 一般USB数据线拔出或打印机关闭，可立刻知道。
 * 蓝牙2.0打印机关闭，大约2s可以知道。
 * TCP网络打印机关闭，大约30s可以知道。
 */
public class CAPrinterConnector {

    private static final String TAG = "PrinterConnector";

    public static final int ConnectionStatus_Disconnected = 0;
    public static final int ConnectionStatus_Connecting = 1;
    public static final int ConnectionStatus_Connected = 2;
    public static final int ConnectionStatus_Disconnecting = 3;

    private CAPrinterDevice currentPrinterDevice = null;
    private Pointer currentPrinterHandle = Pointer.NULL;
    private int currentConnectionStatus = ConnectionStatus_Disconnected;
    private ReentrantLock connectionLock = new ReentrantLock();

    private AutoReplyPrint.CP_OnPortClosedEvent_Callback m_closed_callback = new AutoReplyPrint.CP_OnPortClosedEvent_Callback() {
        @Override
        public void CP_OnPortClosedEvent(Pointer handle, Pointer private_data) {
            if (!handle.equals(currentPrinterHandle))
                return;
            disconnectPrinter();
        }
    };

    public CAPrinterConnector() {
        AutoReplyPrint.INSTANCE.CP_Port_AddOnPortClosedEvent(m_closed_callback, Pointer.NULL);
    }

    protected void finalize() {
        AutoReplyPrint.INSTANCE.CP_Port_RemoveOnPortClosedEvent(m_closed_callback);
        disconnectPrinter();
    }


    private boolean isConnectThreadRunning = false;

    // 启动一个子线程去连接打印机设备（异步），连接结果需要通过 连接状态变化回调接口 获取。
    // 返回值仅表示本次是否启动子线程。
    public boolean connectPrinterAsync(final CAPrinterDevice printerDevice) {
        if (isConnectThreadRunning)
            return false;
        isConnectThreadRunning = true;
        new Thread(new Runnable() {
            @Override
            public void run() {
                connectPrinterSync(printerDevice);
                isConnectThreadRunning = false;
            }
        }).start();
        return true;
    }

    // 直接连接打印机设备（同步），返回连接结果。
    public boolean connectPrinterSync(CAPrinterDevice printerDevice) {
        try {
            connectionLock.lock();
        } catch (Throwable tr) {
            tr.printStackTrace();
        }

        try {
            if ((currentConnectionStatus == ConnectionStatus_Connected) && (!CAPrinterDevice.isPortEqual(currentPrinterDevice, printerDevice))) {
                disconnectPrinter();
            }
        } catch (Throwable tr) {
            tr.printStackTrace();
        }

        if (currentConnectionStatus != ConnectionStatus_Connected) {
            try {
                currentPrinterDevice = printerDevice;
                currentConnectionStatus = ConnectionStatus_Connecting;
                triggerConnectionStatusChangedEvent();
            } catch (Throwable tr) {
                tr.printStackTrace();
            }

            try {
                String m_currentPortType = printerDevice.port_type;
                if (m_currentPortType.compareTo(CAPrinterDevice.PrinterDevicePortTypeCom) == 0) {
                    currentPrinterHandle = AutoReplyPrint.INSTANCE.CP_Port_OpenCom(printerDevice.port_address, printerDevice.comBaudrate, printerDevice.comDataBits, printerDevice.comParity, printerDevice.comStopBits, printerDevice.comFlowControl, printerDevice.autoreplymode);
                } else if (m_currentPortType.compareTo(CAPrinterDevice.PrinterDevicePortTypeUsb) == 0) {
                    currentPrinterHandle = AutoReplyPrint.INSTANCE.CP_Port_OpenUsb(printerDevice.port_address, printerDevice.autoreplymode);
                } else if (m_currentPortType.compareTo(CAPrinterDevice.PrinterDevicePortTypeTcp) == 0) {
                    currentPrinterHandle = AutoReplyPrint.INSTANCE.CP_Port_OpenTcp(null, printerDevice.port_address, (short) printerDevice.tcpPortNumber, 5000, printerDevice.autoreplymode);
                } else if (m_currentPortType.compareTo(CAPrinterDevice.PrinterDevicePortTypeBtSpp) == 0) {
                    currentPrinterHandle = AutoReplyPrint.INSTANCE.CP_Port_OpenBtSpp(printerDevice.port_address, printerDevice.autoreplymode);
                } else if (m_currentPortType.compareTo(CAPrinterDevice.PrinterDevicePortTypeBtBle) == 0) {
                    currentPrinterHandle = AutoReplyPrint.INSTANCE.CP_Port_OpenBtBle(printerDevice.port_address, printerDevice.autoreplymode);
                } else {
                    currentPrinterHandle = Pointer.NULL;
                }
            } catch (Throwable tr) {
                tr.printStackTrace();
            }

            try {
                if (currentPrinterHandle != Pointer.NULL) {
                    currentConnectionStatus = ConnectionStatus_Connected;
                } else {
                    currentConnectionStatus = ConnectionStatus_Disconnected;
                }
                triggerConnectionStatusChangedEvent();
            } catch (Throwable tr) {
                tr.printStackTrace();
            }
        }

        try {
            connectionLock.unlock();
        } catch (Throwable tr) {
            tr.printStackTrace();
        }

        return (currentConnectionStatus == ConnectionStatus_Connected);
    }

    // 断开打印机连接
    public void disconnectPrinter() {
        if (currentPrinterHandle != Pointer.NULL) {
            try {
                connectionLock.lock();
            } catch (Throwable tr) {
                tr.printStackTrace();
            }

            if (currentPrinterHandle != Pointer.NULL) {
                try {
                    currentConnectionStatus = ConnectionStatus_Disconnecting;
                    triggerConnectionStatusChangedEvent();
                } catch (Throwable tr) {
                    tr.printStackTrace();
                }

                try {
                    AutoReplyPrint.INSTANCE.CP_Port_Close(currentPrinterHandle);
                    currentPrinterHandle = Pointer.NULL;
                } catch (Throwable tr) {
                    tr.printStackTrace();
                }

                try {
                    currentConnectionStatus = ConnectionStatus_Disconnected;
                    triggerConnectionStatusChangedEvent();
                } catch (Throwable tr) {
                    tr.printStackTrace();
                }
            }

            try {
                connectionLock.unlock();
            } catch (Throwable tr) {
                tr.printStackTrace();
            }
        }
    }


    // 获取当前打印机设备
    public CAPrinterDevice getCurrentPrinterDevice() {
        return currentPrinterDevice;
    }

    // 获取当前打印机连接句柄
    public Pointer getCurrentPrinterHandle() {
        return currentPrinterHandle;
    }

    // 获取当前连接状态
    public int getCurrentConnectionStatus() {
        return currentConnectionStatus;
    }

    // 是否正在连接打印机
    public boolean isCurrentConnectingPrinter() {
        return currentConnectionStatus == ConnectionStatus_Connecting;
    }

    // 是否已连接打印机
    public boolean isCurrentConnectedPrinter() {
        return currentConnectionStatus == ConnectionStatus_Connected;
    }

    // 是否正在断开打印机
    public boolean isCurrentDisconnectingPrinter() {
        return currentConnectionStatus == ConnectionStatus_Disconnecting;
    }

    // 是否已断开打印机
    public boolean isCurrentDisconnectedPrinter() {
        return currentConnectionStatus == ConnectionStatus_Disconnected;
    }


    // 连接状态变化回调接口
    public interface ConnectionStatusChangedInterface {
        void onConnectionStatusChanged();
    }

    private List<ConnectionStatusChangedInterface> m_connectionStatusChangedInterfaceList = new ArrayList<ConnectionStatusChangedInterface>();

    // 注册连接状态变化回调
    public synchronized void registerConnectionStatusChangedEvent(ConnectionStatusChangedInterface event) {
        if (!m_connectionStatusChangedInterfaceList.contains(event))
            m_connectionStatusChangedInterfaceList.add(event);
    }

    // 移除连接状态变化回调
    public synchronized void unregisterConnectionStatusChangedEvent(ConnectionStatusChangedInterface event) {
        if (m_connectionStatusChangedInterfaceList.contains(event))
            m_connectionStatusChangedInterfaceList.remove(event);
    }

    private void triggerConnectionStatusChangedEvent() {
        for (ConnectionStatusChangedInterface event : m_connectionStatusChangedInterfaceList) {
            event.onConnectionStatusChanged();
        }
    }

}
